//
//  ListColor.swift
//  Do It
//
//  Created by Jim Dovey on 2/14/20.
//  Copyright © 2020 Jim Dovey. All rights reserved.
//

import SwiftUI

enum ListColor: Hashable {
    case red
    case green
    case blue
    case orange
    case yellow
    case pink
    case purple
    case custom(hue: Double, saturation: Double, brightness: Double)
}

extension ListColor {
    var uiColor: SwiftUI.Color {
        switch self {
        case .red: return Color.red
        case .green: return Color.green
        case .blue: return Color.blue
        case .orange: return Color.orange
        case .yellow: return Color.yellow
        case .pink: return Color.pink
        case .purple: return Color.purple
        case let .custom(h, s, b):
            return Color(hue: h, saturation: s, brightness: b)
        }
    }
}

extension ListColor: Codable {
    init(from decoder: Decoder) throws {
        let container = try decoder.singleValueContainer()
        let str = try container.decode(String.self)
        switch str {
        case "red": self = .red
        case "green": self = .green
        case "blue": self = .blue
        case "orange": self = .orange
        case "yellow": self = .yellow
        case "pink": self = .pink
        case "purple": self = .purple

        case _ where str.hasPrefix("hsb:"):
            // An HSB color triple
            let colors = str.dropFirst(4)
                .components(separatedBy: ",")
                .compactMap(Double.init)
                .filter { (0.0...1.0).contains($0) }
            guard colors.count == 3 else {
                throw DecodingError.dataCorruptedError(
                    in: container,
                    debugDescription: "Invalid HSB color value \(str)")
            }
            self = .custom(hue: colors[0], saturation: colors[1], brightness: colors[2])

        default:
            throw DecodingError.dataCorruptedError(
                in: container,
                debugDescription: "Unrecognized color: \"\(str)\"")
        }
    }

    func encode(to encoder: Encoder) throws {
        var container = encoder.singleValueContainer()
        switch self {
        case .red: try container.encode("red")
        case .green: try container.encode("green")
        case .blue: try container.encode("blue")
        case .orange: try container.encode("orange")
        case .yellow: try container.encode("yellow")
        case .pink: try container.encode("pink")
        case .purple: try container.encode("purple")
        case let .custom(h, s, b):
            try container.encode("hsb:\(h),\(s),\(b)")
        }
    }
}

// MARK: - ObjC/CoreData Support

@objc(ListColorClass)
public class ListColorClass: NSObject, NSSecureCoding {
    public static var supportsSecureCoding: Bool { true }

    var color: ListColor

    public override init() {
        self.color = .blue
        super.init()
    }

    init(_ color: ListColor) {
        self.color = color
        super.init()
    }

    public required init?(coder: NSCoder) {
        if let data: NSData = coder.decodeObject(of: NSData.self, forKey: "c") {
            if let color = try? JSONDecoder().decode(ListColor.self, from: data as Data) {
                self.color = color
                return
            }
        }
        self.color = .blue
    }

    public func encode(with coder: NSCoder) {
        do {
            let data = try JSONEncoder().encode(color)
            coder.encode(data, forKey: "c")
        }
        catch {
            preconditionFailure("Error while encoding: \(error)")
        }
    }
}

@objc(SecureColorPermittedArchiver)
class SecureColorPermittedArchiver: NSSecureUnarchiveFromDataTransformer {
    @objc
    override class var allowedTopLevelClasses: [AnyClass] {
        NSSecureUnarchiveFromDataTransformer.allowedTopLevelClasses + [ListColorClass.self]
    }
}
